function [ tout,yout,d,extdelta,n,u_avg,rms_cinitial] = Master_Sim_1d(  )
%Compute symmetric diffusion on tubule/flat space of given geometry
%   

r = input('r of S^r_d:');
d = input('d of S^r_d:');
Radn = input('Number of mesh intervals along sphere height:');
Cyln = input('Number of mesh intervals along cylinder height:');
geometry = input('Is this on a tube or on flat model space. Tube(1) Flat (2):');
Rad  = input('Radius:');
Ht   = input('Height:');
k    = input('Diffusion Coefficient:');
type = input('Is this a Neumann or a Dirichlet BVP. Neumann(1) Dirichlet (2):');
if type  == 1 %Neumann
    cneu = input('Physical Neumann data on the outer rim:');
else         %Dirichlet
    cdir = input('Dirichlet boundary data on the outer rim as coefficient of B-spline at right end:');
end
tstep      = input('Time Increment:');
tfinal     = input('Final Time Sampled:');
cinitial   = input('Initial data as function handle:');
cinitial_orig = cinitial; %For use later

%Construct the mesh on [-r,H] interval
orig_delta = [-1*Rad:Rad/Radn:0 (Ht-0)/Cyln:(Ht-0)/Cyln:Ht]; %Put knots at the 1/Radn and 1/Cyln spacing. 
[extdelta,n] = extpar(orig_delta,r,d);                       %Guaranteed 1-knot at 0.

%Fit cinitial with a spline
t = -1*Rad:(Ht+Rad)/(100*n):Ht;
t=t';
z = arrayfun(cinitial,t);
[cinitial,rms_cinitial] = lsqspl(d,n,extdelta,t,z);


%%%%%%%%%%%% Solve the Tube Neumann and Dirichlet Problems %%%%%%%%%%%%%%%%
%Build the mass and stiffness matrices for the tube case. If Neumann make
%the forcing vector also.
if geometry == 1
    %Note that our Galerkin approximation has the form
    %\sum_j{\int^H_-r N_i*N_j}c'_j + \sum_j{\int^H_-r N'_i*[ ( (r^2-z^2)/r^2)*\chi_[-r,0] + chi_[0,H] ]*N'_j}c_j
    %=N_i(H){\nabla u\cdot\vec{n}}.  Below we also include the k value.
    
    S1 = Assemble_Time(orig_delta,r,d,@(x) 1);                                       %The time component coefficient is 
                                                                                     %\equiv 1 for us                                                                                 
    S2 = Assemble_Space(orig_delta,r,d,@(x) ((Rad^2-x^2)/(Rad^2)).*(x<=0) + (x>0));
    S2 = k*S2;                                                                     %Include diffusion coefficient                                                                                
    if type == 1                                                                   %Neumann Problem so build the linear 
                                                                                   %forcing term.
        In = eye(n);                                                               %For grabbing basis spline coefficients.
        f = zeros(n,1);
        for i=1:n
            f(i,1) = sval(d,extdelta,In(i,:),Ht,findinterval(n,extdelta,Ht))*cneu; %forcing term.
        end
        clear i
        f = k*f;                                                                  %Include diffusion coefficent
    end
end

%Run the ODE diffusion part of tube problem
if geometry == 1 
    if type == 1 %Neumann case
        %S1c' + S2c = f
        DE = @(t,y) S1^(-1)*(-S2*y + f); %ODE to be input in ODE45
        y0 = cinitial;                   %Vector of initial spline coefficients
        [tout,yout] = ode45(DE,0:tstep:tfinal,y0);
    else         %Dirichlet
                 %Because we have a constant right endpoint dirichlet condition, we only test by the
                 %first n-1 basis splines that vanish at that end as cn=cdir Thus the system is
                 %0 = [S1(1,:);...;S1(n-1,:)][c1' ... c(n-1)' 0]' + [S2(1,:);...;S2(n-1,:)][c1 ... c(n-1) cdir]'
                 %This may be simplified to a system for n-1 coefficients like so
                 %0 = S1(1:n-1,1:n-1)[c1' ... c(n-1)']' + S2(1:n-1,1:n-1)[c1 ... c(n-1)] + cdir*S2(1:n-1,n)

        A = S1(1:n-1,1:n-1);
        B = S2(1:n-1,1:n-1);
        f = cdir*S2(1:n-1,n);

        %0 = Ac' + Bc + f
        DE = @(t,y) A^(-1)*(-B*y-f); %ODE equation for input in ODE45
        y0 = cinitial(1,1:n-1)'; %Vector of initial data
        [tout,yout] = ode45(DE,0:tstep:tfinal,y0); 

        %Augment yout to include now the constant cn = cdir term.
        yout = [yout cdir+zeros(size(yout,1),1)];
    end
    %Build the averages and if a Neumann problem compute the errors.
    tcount = size(tout,1);
    u_avg = zeros(tcount,1);
    for i=1:tcount
        u_avg(i,1) = 2*pi*Rad*intspl(d,extdelta,yout(i,:),-Rad,Ht); %\int ud\sigma = \int^(2pi)_(\theta=0)\int^H_(-r) urdzd\theta
    end
    if type == 1 %Neumann problem
        avg_error = u_avg - (cneu*2*pi*Rad*k*tout+u_avg(1,1));
        avg_error = abs(avg_error);
        u_avg = [u_avg avg_error];
    end    
end

%%%%%%%%%%%%% Solve the Flat Case Dirichlet and Neumann Problems %%%%%%%%%%
%Build the mass and stiffness matrices for the flat case. If Neumann make
%the forcing vector also.
if geometry == 2
    %Note that our Galerkin approximation has the form
    %\sum_j{\int^(Ht)_(-Rad) (z+Rad)*N_i*N_jdz}c'_j + \sum_j{\int^(Ht)_(-Rad) N'_i*[ k*(z+Rad) ]*N'_j}c_j
    %=k*N_i(H){\nabla u\cdot\vec{n}}_{phys}*(Rad+Ht).  Below we also include the k value.
    
    %Find the flat geometry effective radii
    Rad_old = Rad;
    Ht_old  = Ht;
    Rad     = sqrt(2)*Rad_old;                                            %pi*r^2_eff = 2*pi*r^2;
    Ht      = sqrt(2*Rad_old*Ht_old+2*Rad_old^2)-sqrt(2)*Rad_old;     %pi*(r_eff+h_eff)^2 - pi*r^2_eff = 2*pi*r*h
    
    %Redraw the mesh on new [-r_eff,H_eff]
    orig_delta = [-1*Rad:Rad/Radn:0 (Ht-0)/Cyln:(Ht-0)/Cyln:Ht]; %Put knots at the 1/Radn and 1/Cyln spacing. 
    [extdelta,n] = extpar(orig_delta,r,d);                       %Guaranteed 1-knot at 0.
    
    %Refit the cinitial on new mesh
    t = -1*Rad:(Ht+Rad)/(100*n):Ht;
    t=t';
    z = arrayfun(cinitial_orig,t);
    [cinitial,rms_cinitial] = lsqspl(d,n,extdelta,t,z);
    

    S1 = Assemble_Time(orig_delta,r,d,@(x) Rad+x);                                 %The time component coefficient is 
                                                                                   %\equiv Rad+x.                                                                                 
    S2 = Assemble_Space(orig_delta,r,d,@(x) Rad+x);
    S2 = k*S2;                                                                     %Include diffusion coefficient                                                                                
    if type == 1                                                                   %Neumann Problem so build the linear 
                                                                                   %forcing term.
        In = eye(n);                                                               %For grabbing basis spline coefficients.
        f = zeros(n,1);
        for i=1:n
            f(i,1) = sval(d,extdelta,In(i,:),Ht,findinterval(n,extdelta,Ht))*cneu*(Rad+Ht); %forcing term.
        end
        clear i
        f = k*f;                                                                  %Include diffusion coefficent
    end
end

%Run the ODE diffusion part of the flat problem
if geometry == 2 
    if type == 1 %Neumann case
        %S1c' + S2c = f
        DE = @(t,y) S1^(-1)*(-S2*y + f); %ODE to be input in ODE45
        y0 = cinitial;                   %Vector of initial spline coefficients
        [tout,yout] = ode45(DE,0:tstep:tfinal,y0);
    else         %Dirichlet
                 %Because we have a constant right endpoint dirichlet condition, we only test by the
                 %first n-1 basis splines that vanish at that end as cn=cdir Thus the system is
                 %0 = [S1(1,:);...;S1(n-1,:)][c1' ... c(n-1)' 0]' + [S2(1,:);...;S2(n-1,:)][c1 ... c(n-1) cdir]'
                 %This may be simplified to a system for n-1 coefficients like so
                 %0 = S1(1:n-1,1:n-1)[c1' ... c(n-1)']' + S2(1:n-1,1:n-1)[c1 ... c(n-1)] + cdir*S2(1:n-1,n)

        A = S1(1:n-1,1:n-1);
        B = S2(1:n-1,1:n-1);
        f = cdir*S2(1:n-1,n);

        %0 = Ac' + Bc + f
        DE = @(t,y) A^(-1)*(-B*y-f); %ODE equation for input in ODE45
        y0 = cinitial(1,1:n-1)'; %Vector of initial data
        [tout,yout] = ode45(DE,0:tstep:tfinal,y0); 

        %Augment yout to include now the constant cn = cdir term.
        yout = [yout cdir+zeros(size(yout,1),1)];
    end
    %Compute the averages for the flat problem
    %Setup the averages column vector
    tcount = size(tout,1);
    u_avg = zeros(tcount,1);
    
    %Extract the weights to be used
    [w,rq,sq] = quadset(6);
    nelm = size(orig_delta,2)-1;
    
    %Build the averages column vector
    for i=1:tcount
        for kt=1:nelm
        %Extract weight sample points
            bk = orig_delta(1,kt+1);
            ak = orig_delta(1,kt);
            sample = ak*rq+bk*sq;
            
        %Evaluate Coefficient and Spline
            valc = arrayfun(@(x) Rad+x,sample);
            valu = sval2(d,extdelta,n,yout(i,:),sample); 
            
            u_avg(i,1) = u_avg(i,1) + (bk-ak)/2*sum(w.*valc.*valu); %Schumaker (1.62)
        end
            u_avg(i,1) = 2*pi*u_avg(i,1);
    end
            if type == 1 %Neumann problem
                avg_error = u_avg - (cneu*2*pi*(Rad+Ht)*k*tout+u_avg(1,1));
                avg_error = abs(avg_error);
                u_avg = [u_avg avg_error];
            end
    end   
end

